%% ----------- [4.5 points] ---------- %%

symbolicOutput(0).  % set to 1 for DEBUGGING: to see symbolic output only; 0 otherwise.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Bricks is a combinatorial number-placement puzzle that uses a partially 
%% filled NxN grid, for 3 <= N <= 9. The objective is to fill the grid
%% with the digits 1 to N, so that each column and each row contain only
%% one of each digit from 1 to N and each brick (corresponding to sublists of
%% length 2 in the input) contains an odd number and an even number.
%%
%% A bricks puzzle alternates rows of type 1 and rows of type 2, which are represented as follows:
%% a) A row of type *1* is represented by a list *starting* with a sublist of length *1*,
%%    followed by sublists of length 2, except the last one (with length 1) if N is even.
%% b) A row of type *2* is represented by a list *starting* with a sublist of length *2*,
%%    followed by sublists of length 2, except the last one (with length 1) if N is odd.
%%
%% Complete the missing Prolog code (indicated with %...) so that given a
%% partially filled bricks puzzle, the program finds a full solution.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%% begin input example "wall1" %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Prolog input:  6x6 grid                               input                      output
% wall( [[[5,-], [1,2], [-,-]],     % type 2    [5   -] [1   2] [-   -]     [5   4] [1   2] [3   6] 
%        [[-],[3,6],[1,-],[-]],     % type 1    [-] [3   6] [1   -] [-]     [4] [3   6] [1   2] [5] 
%        [[-,6], [5,4], [-,2]],     % type 2    [-   6] [5   4] [-   2]     [3   6] [5   4] [1   2]
%        [[6],[-,-],[-,4],[3]],     % type 1    [6] [-   -] [-   4] [3]     [6] [1   2] [5   4] [3]
%        [[-,-], [-,-], [-,4]],     % type 2    [-   -] [-   -] [-   4]     [1   2] [3   6] [5   4]
%        [[2],[5,-],[-,-],[-]]] ).  % type 1    [2] [5   -] [-   -] [-]     [2] [5   4] [3   6] [1]

%%%%%%% end input example wall1 %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%% begin input example "wall2" %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Prolog input:  9x9 grid
%% wall( [ [[3],[-,-],[-,-],[-,-],[9,2]],      % type 1
%%         [[2,1],[-,6],[-,8],[-,4],[-]],      % type 2
%%         [[-],[-,7],[-,3],[9,-],[-,6]],      % type 1
%%         [[6,-],[-,5],[2,-],[7,8],[1]],      % type 2  
%%         [[-],[6,5],[-,-],[-,3],[7,-]],      % type 1
%%         [[-,5],[-,9],[-,-],[1,-],[-]],      % type 2                                
%%         [[-],[4,9],[-,-],[-,-],[-,-]],      % type 1
%%         [[-,-],[-,-],[-,1],[4,-],[-]],      % type 2  
%%         [[-],[-,-],[-,-],[7,2],[-,5]] ] ).  % type 1
%%
%%                 input                                     output
%%  [3] [-   -] [-   -] [-   -] [9   2]        [3] [7   6] [1   8] [4   5] [9   2] 
%%  [2   1] [-   6] [-   8] [-   4] [-]        [2   1] [3   6] [5   8] [9   4] [7] 
%%  [-] [-   7] [-   3] [9   -] [-   6]        [5] [2   7] [4   3] [9   8] [1   6]
%%  [6   -] [-   5] [2   -] [7   8] [1]        [6   9] [4   5] [2   3] [7   8] [1]
%%  [-] [6   5] [-   -] [-   3] [7   -]        [1] [6   5] [8   9] [2   3] [7   4]
%%  [-   5] [-   9] [-   -] [1   -] [-]        [4   5] [8   9] [7   6] [1   2] [3]
%%  [-] [4   9] [-   -] [-   -] [-   -]        [7] [4   9] [2   1] [5   6] [3   8] 
%%  [-   -] [-   -] [-   1] [4   -] [-]        [8   3] [2   7] [6   1] [4   5] [9]
%%  [-] [-   -] [-   -] [7   2] [-   5]        [9] [8   1] [3   4] [7   2] [6   5] 
%%
%%%%%%% end input example wall2 %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%% Some helpful definitions to make the code cleaner: ====================================

size(N) :- wall(W), length(W,N).
row(I)  :- size(N), between(1,N,I).
col(J)  :- size(N), between(1,N,J).
val(K)  :- size(N), between(1,N,K).
odd(X)  :- 1 is X mod 2.
even(X) :- 0 is X mod 2.

% row I *starts* with a non-brick (a sublist of length 1)
type1row(I) :- 
        wall(W), row(I), nth1(I,W,Row),
        nth1(1,Row,E), length(E,1).

% row I *starts* with a brick (a sublist of length 2)
type2row(I) :- 
        wall(W), row(I), nth1(I,W,Row),
        nth1(1,Row,E), length(E,2).

% (Iid,Jid) is a brick ID (defined as the left cell coordinates of some brick)
% For instance, with the previous example "wall1", these brick IDs will be obtained:
% ?- findall((Iid,Jid), brickID(Iid,Jid), Lid), sort(Lid, Lid1).
% Lid1 = [(1,1),(1,3),(1,5),(2,2),(2,4),(3,1),(3,3),(3,5),(4,2),(4,4),(5,1),(5,3),(5,5),(6,2),(6,4)]
brickID(Iid, Jid) :-
        row(Iid), type1row(Iid),
        col(Jid), even(Jid),         
        size(N), Jid \= N.
brickID(Iid, Jid) :-
        row(Iid), type2row(Iid),
        col(Jid), odd(Jid), 
        size(N), Jid \= N.

% (I,J) is a cell of a brick whose ID is (Iid,Jid)
% For instance, with brickID (2,4):
% ?- findall((I,J), squareOfBrick(2,4,I,J), L).
% L = [(2,4),(2,5)].
squareOfBrick(Iid,Jid,I,J) :- I = Iid, J1 is Jid+1, between(Jid,J1,J).

%%%%%%% End helpful definitions ===============================================================


%%%%%%%  1. SAT Variables: ====================================================================

% value(I,J,K) means "square IJ gets value K",    1 <= I <= N, 1 <= J <= N, 1 <=  K <= N
satVariable( cell(I,J,K) ) :- row(I), col(J), val(K).


%%%%%%%  2. Clause generation for the SAT solver: =============================================

writeClauses :- 
    filledInputValues,         % for each filled-in value of the input, add a unit clause
    eachIJexactlyOneK,         % each square IJ gets exactly one value K
    eachJKexactlyOneI,         % each column J gets each value K in exactly one row I
    eachIKexactlyOneJ,         % each row    I gets each value K in exactly one column J
    eachBrickContainsOdd,      % each brick contains an odd number
    eachBrickContainsEven,     % each brick contains an even number
    true,!.                    % this way you can comment out ANY previous line of writeClauses
writeClauses :- told, nl, write('writeClauses failed!'), nl,nl, halt.

filledInputValues :-      %% HINT: use flatten/2
        wall(W), nth1(I,W,Row), 
        flatten(Row,R), nth1(J,R,K),
        integer(K), writeOneClause([cell(I,J,K)]), fail.
filledInputValues.

eachIJexactlyOneK :- 
        row(I), col(J), 
        findall(cell(I,J,K),val(K),Lits), 
        exactly(1,Lits), fail.
eachIJexactlyOneK.

eachJKexactlyOneI :- 
        col(J), val(K), 
        findall(cell(I,J,K),row(I),Lits), 
        exactly(1,Lits), fail.
eachJKexactlyOneI.

eachIKexactlyOneJ :- 
        row(I), val(K), 
        findall(cell(I,J,K),col(J),Lits), 
        exactly(1,Lits), fail.
eachIKexactlyOneJ.

eachBrickContainsOdd :- 
        brickID(Iid,Jid),
        size(N), findall(X, (between(1,N,X), odd(X)), Odd),
        findall(cell(I,J,K), (squareOfBrick(Iid,Jid,I,J), member(K,Odd)), Lits), 
        exactly(1,Lits), fail.
eachBrickContainsOdd.

eachBrickContainsEven :- 
        brickID(Iid,Jid),
        size(N), findall(X, (between(1,N,X), even(X)), Even),
        findall(cell(I,J,K), (squareOfBrick(Iid,Jid,I,J), member(K,Even)), Lits), 
        exactly(1,Lits), fail.
eachBrickContainsEven.


%%%%%%%  3. DisplaySol: show the solution. Here M contains the literals that are true in the model:

% displaySol(M) :- nl, write(M), nl, nl, fail.
displaySol(M) :- nl, row(I), nl, col(J), member(cell(I,J,K), M ), displayCell(I,J,K), write(' '), fail.
displaySol(_) :- nl, nl.

displayCell(I,J,K) :- 
        type1row(I), even(J), size(N), J \= N, write('['), write(K), write('  ').
displayCell(I,J,K) :- 
        type1row(I), odd(J), J \= 1, write(K), write(']').        
displayCell(I,J,K) :-
        type1row(I), J = 1, write('['), write(K), write(']').
displayCell(I,J,K) :-
        type1row(I), size(N), even(N), J = N, write('['), write(K), write(']').
displayCell(I,J,K) :-
        type2row(I), odd(J), size(N), J \= N, write('['), write(K), write('  ').
displayCell(I,J,K) :- 
        type2row(I), even(J), write(K), write(']').
displayCell(I,J,K) :-
        type2row(I), odd(J), size(N), J = N, write('['), write(K), write(']').

%%%%%%% =======================================================================================



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Everything below is given as a standard library, reusable for solving
%%    with SAT many different problems.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%% Cardinality constraints on arbitrary sets of literals Lits: ===========================

exactly(K,Lits) :- symbolicOutput(1), write( exactly(K,Lits) ), nl, !.
exactly(K,Lits) :- atLeast(K,Lits), atMost(K,Lits),!.

atMost(K,Lits) :- symbolicOutput(1), write( atMost(K,Lits) ), nl, !.
atMost(K,Lits) :-   % l1+...+ln <= k:  in all subsets of size k+1, at least one is false:
      negateAll(Lits,NLits),
      K1 is K+1,    subsetOfSize(K1,NLits,Clause), writeOneClause(Clause), fail.
atMost(_,_).

atLeast(K,Lits) :- symbolicOutput(1), write( atLeast(K,Lits) ), nl, !.
atLeast(K,Lits) :-  % l1+...+ln >= k: in all subsets of size n-k+1, at least one is true:
      length(Lits,N),
      K1 is N-K+1,  subsetOfSize(K1, Lits,Clause), writeOneClause(Clause), fail.
atLeast(_,_).

negateAll([], []).
negateAll([Lit|Lits], [NLit|NLits]) :- negate(Lit,NLit), negateAll(Lits, NLits), !.

negate(-Var,  Var) :- !.
negate( Var, -Var) :- !.

subsetOfSize(0,_,[]) :- !.
subsetOfSize(N,[X|L],[X|S]) :- N1 is N-1, length(L,Leng), Leng>=N1, subsetOfSize(N1,L,S).
subsetOfSize(N,[_|L],   S ) :-            length(L,Leng), Leng>=N,  subsetOfSize( N,L,S).


%%%%%%% Express equivalence between a variable and a disjunction or conjunction of literals ===

% Express that Var is equivalent to the disjunction of Lits:
expressOr(Var, Lits) :- symbolicOutput(1), write( Var ), write(' <--> or('), write(Lits), write(')'), nl, !.
expressOr(Var, Lits) :- member(Lit,Lits), negate(Lit,NLit), writeOneClause([NLit,Var]), fail.
expressOr(Var, Lits) :- negate(Var,NVar), writeOneClause([NVar|Lits]),!.

%% expressOr(a,[x,y]) genera 3 clausulas (como en la Transformación de Tseitin):
%% a == x v y
%% x -> a       -x v a
%% y -> a       -y v a
%% a -> x v y   -a v x v y

% Express that Var is equivalent to the conjunction of Lits:
expressAnd(Var, Lits) :- symbolicOutput(1), write( Var ), write(' <--> and('), write(Lits), write(')'), nl, !.
expressAnd(Var, Lits) :- member(Lit,Lits), negate(Var,NVar), writeOneClause([NVar,Lit]), fail.
expressAnd(Var, Lits) :- findall(NLit, (member(Lit,Lits), negate(Lit,NLit)), NLits), writeOneClause([Var|NLits]), !.


%%%%%%% main: =================================================================================

main:-  current_prolog_flag(os_argv, Argv),
        nth0(1, Argv, InputFile),
        main(InputFile), !.
main:-  write('Usage: $ ./<executable> <example>          or ?- main(<example>).'), nl, halt.

main(InputFile):-
        symbolicOutput(1), !,
        consult(InputFile),
        writeClauses, halt.   % print the clauses in symbolic form and halt Prolog
main(InputFile):-
        consult(InputFile),
        initClauseGeneration,
        tell(clauses), writeClauses, told,          % generate the (numeric) SAT clauses and call the solver
        tell(header),  writeHeader,  told,
        numVars(N), numClauses(C),
        write('Generated '), write(C), write(' clauses over '), write(N), write(' variables. '),nl,
        shell('cat header clauses > infile.cnf',_),
        write('Calling solver....'), nl,
        shell('kissat -v infile.cnf > model', Result),  % if sat: Result=10; if unsat: Result=20.
        treatResult(Result),!.

treatResult(20) :- write('Unsatisfiable'), nl, halt.
treatResult(10) :- write('Solution found: '), nl, see(model), symbolicModel(M), seen, displaySol(M), nl,nl,halt.
treatResult( _) :- write('cnf input error. Wrote anything strange in your cnf?'), nl,nl, halt.


initClauseGeneration:-  %initialize all info about variables and clauses:
        retractall(numClauses(   _)),
        retractall(numVars(      _)),
        retractall(varNumber(_,_,_)),
        assert(numClauses( 0 )),
        assert(numVars(    0 )),     !.

writeOneClause([]) :- symbolicOutput(1),!, nl.
writeOneClause([]) :- countClause, write(0), nl.
writeOneClause([Lit|C]) :- w(Lit), writeOneClause(C),!.
w(-Var) :- symbolicOutput(1), satVariable(Var), write(-Var), write(' '),!.
w( Var) :- symbolicOutput(1), satVariable(Var), write( Var), write(' '),!.
w(-Var) :- satVariable(Var),  var2num(Var,N),   write(-), write(N), write(' '),!.
w( Var) :- satVariable(Var),  var2num(Var,N),             write(N), write(' '),!.
w( Lit) :- told, write('ERROR: generating clause with undeclared variable in literal '), write(Lit), nl,nl, halt.


% given the symbolic variable V, find its variable number N in the SAT solver:
:- dynamic(varNumber / 3).
var2num(V,N) :- hash_term(V,Key), existsOrCreate(V,Key,N),!.
existsOrCreate(V,Key,N) :- varNumber(Key,V,N),!.                            % V already existed with num N
existsOrCreate(V,Key,N) :- newVarNumber(N), assert(varNumber(Key,V,N)), !.  % otherwise, introduce new N for V

writeHeader :- numVars(N),numClauses(C), write('p cnf '),write(N), write(' '),write(C),nl.

countClause :-     retract( numClauses(N0) ), N is N0+1, assert( numClauses(N) ),!.
newVarNumber(N) :- retract( numVars(   N0) ), N is N0+1, assert(    numVars(N) ),!.

% Getting the symbolic model M from the output file:
symbolicModel(M) :- get_code(Char), readWord(Char,W), symbolicModel(M1), addIfPositiveInt(W,M1,M),!.
symbolicModel([]).
addIfPositiveInt(W,L,[Var|L]) :- W = [C|_], between(48,57,C), number_codes(N,W), N>0, varNumber(_,Var,N),!.
addIfPositiveInt(_,L,L).
readWord( 99,W) :- repeat, get_code(Ch), member(Ch,[-1,10]), !, get_code(Ch1), readWord(Ch1,W),!. % skip line starting w/ c
readWord(115,W) :- repeat, get_code(Ch), member(Ch,[-1,10]), !, get_code(Ch1), readWord(Ch1,W),!. % skip line starting w/ s
readWord( -1,_) :-!, fail. %end of file
readWord(C, []) :- member(C,[10,32]), !. % newline or white space marks end of word
readWord(Char,[Char|W]) :- get_code(Char1), readWord(Char1,W), !.

%%%%%%% =======================================================================================
